/*
    FreeRTOS V7.0.2 - Copyright (C) 2011 Real Time Engineers Ltd.
	

    ***************************************************************************
     *                                                                       *
     *    FreeRTOS tutorial books are available in pdf and paperback.        *
     *    Complete, revised, and edited pdf reference manuals are also       *
     *    available.                                                         *
     *                                                                       *
     *    Purchasing FreeRTOS documentation will not only help you, by       *
     *    ensuring you get running as quickly as possible and with an        *
     *    in-depth knowledge of how to use FreeRTOS, it will also help       *
     *    the FreeRTOS project to continue with its mission of providing     *
     *    professional grade, cross platform, de facto standard solutions    *
     *    for microcontrollers - completely free of charge!                  *
     *                                                                       *
     *    >>> See http://www.FreeRTOS.org/Documentation for details. <<<     *
     *                                                                       *
     *    Thank you for using FreeRTOS, and thank you for your support!      *
     *                                                                       *
    ***************************************************************************


    This file is part of the FreeRTOS distribution.

    FreeRTOS is free software; you can redistribute it and/or modify it under
    the terms of the GNU General Public License (version 2) as published by the
    Free Software Foundation AND MODIFIED BY the FreeRTOS exception.
    >>>NOTE<<< The modification to the GPL is included to allow you to
    distribute a combined work that includes FreeRTOS without being obliged to
    provide the source code for proprietary components outside of the FreeRTOS
    kernel.  FreeRTOS is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details. You should have received a copy of the GNU General Public
    License and the FreeRTOS license exception along with FreeRTOS; if not it
    can be viewed here: http://www.freertos.org/a00114.html and also obtained
    by writing to Richard Barry, contact details for whom are available on the
    FreeRTOS WEB site.

    1 tab == 4 spaces!

    http://www.FreeRTOS.org - Documentation, latest information, license and
    contact details.

    http://www.SafeRTOS.com - A version that is certified for use in safety
    critical systems.

    http://www.OpenRTOS.com - Commercial support, development, porting,
    licensing and training services.
*/


/* Standard includes. */
#include <stdlib.h>
#include <stdio.h>
/* Scheduler includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "wm_config.h"

#if TLS_OS_FREERTOS

#ifndef configMAX_SYSCALL_INTERRUPT_PRIORITY
#define configMAX_SYSCALL_INTERRUPT_PRIORITY 	191 /* equivalent to 0xb0, or priority 11. */
#endif

#ifndef configKERNEL_INTERRUPT_PRIORITY
	#define configKERNEL_INTERRUPT_PRIORITY 255
#endif
/* Constants required to manipulate the NVIC. */
#define portNVIC_SYSTICK_CTRL		( ( volatile unsigned long *) 0xe000e010 )
#define portNVIC_SYSTICK_LOAD		( ( volatile unsigned long *) 0xe000e014 )
#define portNVIC_INT_CTRL			( ( volatile unsigned long *) 0xe000ed04 )
#define portNVIC_SYSPRI2			( ( volatile unsigned long *) 0xe000ed20 )
#define portNVIC_SYSTICK_CLK		0x00000004
#define portNVIC_SYSTICK_INT		0x00000002
#define portNVIC_SYSTICK_ENABLE		0x00000001
#define portNVIC_PENDSVSET			0x10000000
#define portNVIC_PENDSV_PRI			( ( ( unsigned long ) configKERNEL_INTERRUPT_PRIORITY ) << 16 )
#define portNVIC_SYSTICK_PRI		( ( ( unsigned long ) configKERNEL_INTERRUPT_PRIORITY ) << 24 )

/* Constants required to set up the initial stack. */
#define portINITIAL_XPSR			( 0x01000000 )


/*-----------------------------------------------------------*/

/* The code generated by the Keil compiler does not maintain separate
stack and frame pointers. The portENTER_CRITICAL macro cannot therefore
use the stack as per other ports.  Instead a variable is used to keep
track of the critical section nesting.  This variable has to be stored
as part of the task context and must be initialised to a non zero value. */

#define portNO_CRITICAL_NESTING		( ( unsigned portLONG ) 0 )
volatile unsigned portLONG ulCriticalNesting = 9999UL;

/*-----------------------------------------------------------*/

/* Setup the timer to generate the tick interrupts. */
//static void prvSetupTimerInterrupt( void );

/* 
 * The scheduler can only be started from ARM mode, so 
 * vPortStartFirstSTask() is defined in portISR.c. 
 */
extern void vPortStartFirstTask( void );

/* Each task maintains its own interrupt status in the critical nesting
variable. */
static unsigned portBASE_TYPE uxCriticalNesting = 0xaaaaaaaa;

/* 
 * Setup the timer to generate the tick interrupts.
 */
//static void prvSetupTimerInterrupt( void );

/*
 * Exception handlers.
 */
void xPortPendSVHandler( void );
void xPortSysTickHandler( void );
void vPortSVCHandler( void );

/*
 * Start first task is a separate function so it can be tested in isolation.
 */
void vPortStartFirstTask( void );

/*-----------------------------------------------------------*/

/* 
 * See header file for description. 
 */
portSTACK_TYPE *pxPortInitialiseStack( portSTACK_TYPE *pxTopOfStack, pdTASK_CODE pxCode, void *pvParameters )
{
	/* Simulate the stack frame as it would be created by a context switch
	interrupt. */
	pxTopOfStack--; /* Offset added to account for the way the MCU uses the stack on entry/exit of interrupts. */
	*pxTopOfStack = portINITIAL_XPSR;	/* xPSR */
	pxTopOfStack--;
	*pxTopOfStack = ( portSTACK_TYPE ) pxCode;	/* PC */
	pxTopOfStack--;
	*pxTopOfStack = 0;	/* LR */
	pxTopOfStack -= 5;	/* R12, R3, R2 and R1. */
	*pxTopOfStack = ( portSTACK_TYPE ) pvParameters;	/* R0 */
	pxTopOfStack -= 8;	/* R11, R10, R9, R8, R7, R6, R5 and R4. */

	return pxTopOfStack;
}
/*-----------------------------------------------------------*/


__attribute__ ((naked)) void SVC_Handler( void )
{	
	__asm volatile (
					" ldr	r3, =pxCurrentTCB	\n"				/* Restore the context. */
					" ldr r1, [r3]				\n"				/* Use pxCurrentTCBConst to get the pxCurrentTCB address. */
					" ldr r0, [r1]				\n"				/* The first item in pxCurrentTCB is the task top of stack. */
					" ldmia r0!, {r4-r11}		\n"				/* Pop the registers that are not automatically saved on exception entry and the critical nesting count. */
					" msr psp, r0				\n"				/* Restore the task stack pointer. */
					" mov r0, #0			    \n"
					" msr	basepri, r0         \n"
					" orr r14, #0xd				\n"	
					" bx r14	                \n"	
					" 							\n"	
				);						
}
/*-----------------------------------------------------------*/

__attribute__ ((naked)) void vPortStartFirstTask( void )
{	
	__asm volatile (			
					" ldr r0, =0xE000ED08	 \n"					/* Use the NVIC offset register to locate the stack. */
					" ldr r0, [r0]           \n"
					" ldr r0, [r0]           \n"
					"                        \n"
					" msr msp, r0			 \n"					/* Set the msp back to the start of the stack. */
					"                        \n"
					" cpsie i				 \n"					/* Globally enable interrupts. */
					"                        \n"
					" svc 0					 \n"				/* Call SVC to start the first task. */
					" nop                    \n"
				);
}
/*-----------------------------------------------------------*/


int ulSetInterruptMaskFromISR( void )
{
    portDISABLE_INTERRUPTS();
    return 0;
}

void vClearInterruptMaskFromISR( int ulMask )
{
    portENABLE_INTERRUPTS();
}

/* 
 * See header file for description. 
 */
portBASE_TYPE xPortStartScheduler( void )
{
	/* Make PendSV, CallSV and SysTick the same priroity as the kernel. */
	*(portNVIC_SYSPRI2) |= portNVIC_PENDSV_PRI;
#if 0	/*Tick move to wm_main.c*/
	*(portNVIC_SYSPRI2) |= portNVIC_SYSTICK_PRI;

	/* Start the timer that generates the tick ISR.  Interrupts are disabled
	here already. */
	prvSetupTimerInterrupt();
#endif
	/* Initialise the critical nesting count ready for the first task. */
	uxCriticalNesting = 0;

	/* Start the first task. */
	vPortStartFirstTask();

	/* Should not get here! */
	return 0;
}
/*-----------------------------------------------------------*/

void vPortEndScheduler( void )
{
	/* It is unlikely that the CM3 port will require this function as there
	is nothing to return to.  */
}
/*-----------------------------------------------------------*/

void vPortYield( void )
{
	/* Set a PendSV to request a context switch. */
	*(portNVIC_INT_CTRL) = portNVIC_PENDSVSET;
}
/*-----------------------------------------------------------*/

void vPortEnterCritical( void )
{
	portDISABLE_INTERRUPTS();
	uxCriticalNesting++;
}
/*-----------------------------------------------------------*/

void vPortExitCritical( void )
{
	if (uxCriticalNesting)
		uxCriticalNesting--;
	if( uxCriticalNesting == 0 )
	{
		portENABLE_INTERRUPTS();
	}
}
/*-----------------------------------------------------------*/

__attribute__ ((naked)) void PendSV_Handler( void )
{
	__asm volatile(				
					" CPSID   I								\n"
					" mrs r0, psp                           \n"
					" ldr	r3, =pxCurrentTCB  		 	 	\n"			/* Get the location of the current TCB. */
					" ldr	r2, [r3]                        \n"
					" stmdb r0!, {r4-r11}				 	\n"			/* Save the remaining registers. */
					" str r0, [r2]					 		\n"			/* Save the new top of stack into the first member of the TCB. */
					" stmdb sp!, {r3, r14}		            \n"
					" bl vTaskSwitchContext                 \n"
					" ldmia sp!, {r3, r14}	                \n"
					" ldr r1, [r3]					        \n"
					" ldr r0, [r1]					 		\n"	/* The first item in pxCurrentTCB is the task top of stack. */
					" ldmia r0!, {r4-r11}			 		\n"	/* Pop the registers and the critical nesting count. */
					" msr psp, r0	                        \n"
					" CPSIE I                               \n"
					" bx r14                                \n"
					" nop                                   \n"
				);
	
}
/*-----------------------------------------------------------*/

void OS_CPU_SysTickHandler( void )
{
unsigned long ulDummy;
	/* If using preemption, also force a context switch. */
	#if configUSE_PREEMPTION == 1
		*(portNVIC_INT_CTRL) = portNVIC_PENDSVSET;	
	#endif

	ulDummy = portSET_INTERRUPT_MASK_FROM_ISR();
	{
		vTaskIncrementTick();
	}
	portCLEAR_INTERRUPT_MASK_FROM_ISR( ulDummy );
}
/*-----------------------------------------------------------*/

/*
 * Setup the systick timer to generate the tick interrupts at the required
 * frequency.
 */
 #if 0
void prvSetupTimerInterrupt( void )
{
	/* Configure SysTick to interrupt at the requested rate. */
	*(portNVIC_SYSTICK_LOAD) = ( configCPU_CLOCK_HZ / configTICK_RATE_HZ ) - 1UL;
	*(portNVIC_SYSTICK_CTRL) = portNVIC_SYSTICK_CLK | portNVIC_SYSTICK_INT | portNVIC_SYSTICK_ENABLE;
}
#endif
/*-----------------------------------------------------------*/

__attribute__ ((naked)) void portDISABLE_INTERRUPTS( void )
{
	__asm volatile(
					"push { r0 }				 \n"
					"mrs r0,PRIMASK              \n"
					"CPSID I                     \n"
					"pop { r0 }                  \n"
					"bx r14                      \n"
				);
}

/*-----------------------------------------------------------*/

__attribute__ ((naked)) void portENABLE_INTERRUPTS( void )
{
	__asm volatile (
					" push { r0 }			\n"
					" MRS r0,PRIMASK        \n"
					" CPSIE I               \n"
					" pop { r0 }            \n"
					" bx r14                \n"
				);
}

int portGET_IPSR(void)
{
	int result;
	
  __asm volatile ("MRS %0, ipsr" : "=r" (result) );
  
  return (result);
}

void vApplicationStackOverflowHook( xTaskHandle *pxTask, signed char *pcTaskame, signed char prio )
{
	printf("\ntask[%s] prio[%d]stack over flow\n", pcTaskame, prio);
}

#endif


